#include "CommerceSony.h"
#include "SignInSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"


using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;


namespace UnityPlugin
{
	CachedCommerce gCommerce;

	DO_EXPORT( bool, PrxCommerceIsBusy ) ()
	{
		return gCommerce.IsBusy();
	}

	DO_EXPORT( bool, PrxCommerceGetLastError) (ResultCode* result)
	{
		return gCommerce.GetLastError(result);

	}

	DO_EXPORT( ErrorCode, PrxCommerceCreateSession ) ()
	{
		return gCommerce.CreateSession();
	}

	DO_EXPORT( ErrorCode, PrxCommerceBrowseCategory ) (const char* categoryId)
	{
		return gCommerce.BrowseCategory(categoryId);
	}

	DO_EXPORT( ErrorCode, PrxCommerceBrowseProduct ) (const char* productId)
	{
		return gCommerce.BrowseProduct(productId);
	}

	DO_EXPORT( ErrorCode, PrxCommerceDisplayDownloadList ) ()
	{
		return gCommerce.DisplayDownloadList();
	}

	DO_EXPORT( ErrorCode, PrxCommerceCheckout ) (char** skuIDs, int count)
	{
		return gCommerce.Checkout(skuIDs, count);
	}

	DO_EXPORT( ErrorCode, PrxCommerceVoucherInput ) ()
	{
		return gCommerce.VoucherInput();
	}

	DO_EXPORT( ErrorCode, PrxCommerceRequestCategoryInfo ) (const char* categoryId)
	{
		return gCommerce.RequestCategoryInfo(categoryId);
	}

	DO_EXPORT( ErrorCode, PrxCommerceGetCategoryInfo ) (CommerceCategoryInfo* categoryInfo)
	{
		return gCommerce.GetCategoryInfo(categoryInfo);
	}

	DO_EXPORT( ErrorCode, PrxCommerceGetSubCategoryInfo ) (UInt32 index, CommerceCategoryInfo* categoryInfo)
	{
		return gCommerce.GetSubCategoryInfo(index, categoryInfo);
	}

	DO_EXPORT( ErrorCode, PrxCommerceRequestProductList ) (const char* categoryId)
	{
		return gCommerce.RequestProductList(categoryId);
	}

	DO_EXPORT( int, PrxCommerceGetProductListInfoCount ) ()
	{
		return gCommerce.GetProductListInfoCount();
	}

	DO_EXPORT( ErrorCode, PrxCommerceGetProductListInfoItem ) (UInt32 index, CommerceProductInfo* productInfo)
	{
		return gCommerce.GetProductListInfoItem(index, productInfo);
	}

	DO_EXPORT( ErrorCode, PrxCommerceRequestDetailedProductInfo ) (const char* productId)
	{
		return gCommerce.RequestDetailedProductInfo(productId);
	}

	DO_EXPORT( ErrorCode, PrxCommerceGetProductInfoDetailed ) (CommerceProductInfoDetailed* productInfo)
	{
		return gCommerce.GetProductInfoDetailed(productInfo);
	}

	DO_EXPORT( ErrorCode, PrxCommerceRequestEntitlementList ) ()
	{
		return gCommerce.RequestEntitlementList();
	}

	DO_EXPORT( int, PrxCommerceGetEntitlementListCount ) ()
	{
		return gCommerce.GetEntitlementListCount();
	}

	DO_EXPORT( ErrorCode, PrxCommerceGetEntitlementListItem ) (UInt32 index, CommerceEntitlement* entitlement)
	{
		return gCommerce.GetEntitlementListItem(index, entitlement);
	}

	DO_EXPORT( ErrorCode, PrxCommerceConsumeEntitlement ) (const char* entitlementId, int consumeCount)
	{
		return gCommerce.ConsumeEntitlement(entitlementId, consumeCount);
	}

	DO_EXPORT( bool, PrxCommerceShowStoreIcon ) (bool show, int position)
	{
		int ret = -1;

		// Can't find an API for this on PS3 so I guess it's not supported.

		return ret == SCE_OK;
	}

	CachedCommerce::CachedCommerce()
		: m_Busy(false)
		, m_SessionCreated(false)
		,m_CheckoutDlgMemContainer(SYS_MEMORY_CONTAINER_ID_INVALID)
		,m_DlListDlgMemContainer(SYS_MEMORY_CONTAINER_ID_INVALID)
		,m_BrowseProductDlgMemContainer(SYS_MEMORY_CONTAINER_ID_INVALID)
		,m_VoucherInputDlgMemContainer(SYS_MEMORY_CONTAINER_ID_INVALID)
		, m_LastResult("Commerce")
	{
	}

	CachedCommerce::~CachedCommerce()
	{
	}

	bool CachedCommerce::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
			case Event::commerceSessionCreated:				// An event generated when a commerce session has successfully been created.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceSessionCreated);
				break;

			case Event::commerceSessionAborted:				// An event generated when the creation of case commerce session has been aborted.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceSessionAborted);
				break;


			case Event::commerceGotEntitlementList:			// An event generated when the list of entitlements has been received for the current user.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceGotEntitlementList);
				break;

			case Event::commerceConsumedEntitlement:		// An event generated when the user has successfully consumed an entitlement.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceConsumedEntitlement);
				break;

			case Event::commerceGotCategoryInfo:			// An event generated when some category information has been retrieved from the store.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceGotCategoryInfo);
				break;

			case Event::commerceGotProductList:				// An event generated when a list of products that are available has been retrieved from the store.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceGotProductList);
				break;

			case Event::commerceGotDetailedProductInfo:		// An event generated when some detailed product information has been retrieved from the store.
				m_Busy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_CommerceGotProductInfo);
				break;

			case Event::commerceError:						// An event generated when a commerce error has occurred.
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_CommerceError);
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceCheckoutStarted:			// An event generated when a store checkout overlay has started.
				Messages::AddMessage(Messages::kNPToolKit_CommerceCheckoutStarted);
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceCheckoutSuccess:			// An event generated when user has successfully purchased from the checkout.
				CleanupDialogMem();
				Messages::AddMessage(Messages::kNPToolKit_CommerceCheckoutFinished);
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceCheckoutAborted:	// An event generated when the checkout was aborted by the user (user pressed back).
				CleanupDialogMem();
				Messages::AddMessage(Messages::kNPToolKit_CommerceCheckoutFinished);
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceCheckoutFinished:			// An event generated when a store checkout overlay has finished.
				CleanupDialogMem();
				Messages::AddMessage(Messages::kNPToolKit_CommerceCheckoutFinished);
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceProductBrowseStarted:		// An event generated when product overlay has started.
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceProductBrowseSuccess:		// An event generated when a product browse was completed successfully: and the user purchased the product.
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceProductBrowseAborted:		// An event generated when a product browse was aborted by the user (the user pressed back).
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceProductBrowseFinished:		// An event generated when a product browse has finished and it is now safe to free memory.
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceVoucherInputStarted:		// An event generated when a voucher code input overlay was started.
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceVoucherInputSuccess:		// An event generated when a voucher code input completed successfully.
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceVoucherInputAborted:		// An event generated when a voucher code input was aborted by the user (user pressed back).
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceVoucherInputFinished:		// An event generated when a voucher code input has finished. It is now safe to free memory.
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;


			case Event::commerceDownloadListStarted:		// An event generated when a download list overlay has started.
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceDownloadListSuccess:		// An event generated when the user has ended the download list.
				m_Busy = false;
				handled = true;
				break;

			case Event::commerceDownloadListFinished:		// An event generated when a download list overlay has finished.
				CleanupDialogMem();
				m_Busy = false;
				handled = true;
				break;

			// TODO: Are these used ?
			case Event::authGotTicket:
				// Fires when getting entitlements.
				//DBG_LOG_WARNING("Commerce::%s@L%d - does authGotTicket need handling", __FUNCTION__, __LINE__);
				//m_Busy = false;
				handled = true;
				break;

			case Event::authNewTicket:
				DBG_LOG_WARNING("Commerce::%s@L%d - does authNewTicket need handling", __FUNCTION__, __LINE__);
				m_Busy = false;
				handled = true;
				break;

			case Event::authError:
				// getEntitlementList failed.
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_CommerceError);
				m_Busy = false;
				handled = true;
				break;


			case Event::serviceError:
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_CommerceError);
				m_Busy = false;
				handled = true;
				break;

			default:
				break;
		}

		return handled;
	}


	bool CachedCommerce::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_Busy;
	}

	void CachedCommerce::Update()
	{
		// TODO: Probably don't need this.
	}

	ErrorCode CachedCommerce::CreateSession()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(m_SessionCreated)
		{
			return m_LastResult.GetResult();
		}

		int ret = sce::Toolkit::NP::Commerce::Interface::createSession();
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		m_SessionCreated = true;
		return m_LastResult.GetResult();
	}

	// Launches the PlayStation®Store to a specified category where the user can browse and purchase.
	// This causes the application to terminate. When the user has quit from the store, the application will restart automatically.
	// categoryId - The PlayStation®Store category ID. Specify NULL to launch to the root category.
	ErrorCode CachedCommerce::BrowseCategory(const char* categoryId)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		if(categoryId && (strlen(categoryId) == 0))
		{
			categoryId = NULL;
		}

		m_LastResult.Reset();

		int userData = 0;
		int ret = Commerce::Interface::categoryBrowse(categoryId, userData);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		return m_LastResult.GetResult();
	}

	// Launches the PlayStation®Store to a specified product where the user can purchase it.
	// This can be either an in-game store overlay or the launch of the store after the termination of the application. In the latter case, the application will re-start automatically.
	ErrorCode CachedCommerce::BrowseProduct(const char* productId)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		sce::Toolkit::NP::ProductBrowseParams params;
		int ret;
		strncpy(params.productId, productId, sizeof(params.productId));
		params.inGame = true;
		ret = sys_memory_container_create(&m_BrowseProductDlgMemContainer, SCE_NP_COMMERCE2_DO_PROD_BROWSE_MEMORY_CONTAINER_SIZE);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		ret = Commerce::Interface::productBrowse(params, false);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::DisplayDownloadList()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		m_LastResult.Reset();

		DownloadListInputParams params;
		int ret;

		ret = sys_memory_container_create(&m_DlListDlgMemContainer, SCE_NP_COMMERCE2_DO_PROD_BROWSE_MEMORY_CONTAINER_SIZE);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		params.memContainer = &m_DlListDlgMemContainer;
		ret = Commerce::Interface::displayDownloadList(params, false);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::Checkout(char** skuIDs, int count)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		m_LastResult.Reset();

		if(count > 0)
		{
			CheckoutInputParams params;
			int ret;

			ret = sys_memory_container_create(&m_CheckoutDlgMemContainer, SCE_NP_COMMERCE2_DO_CHECKOUT_MEMORY_CONTAINER_SIZE);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}
			params.memContainer = &m_CheckoutDlgMemContainer;

			for(int i=0; i<count; i++)
			{
				params.skuIds.push_back(skuIDs[i]);
			}

			ret = Commerce::Interface::checkout(params, false);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}
			m_Busy = true;
		}
		else
		{
			Messages::LogWarning("Commerce::%s@L%d - %s", __FUNCTION__, __LINE__, "Nothing to checkout");
		}

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::VoucherInput()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		m_LastResult.Reset();

		VoucherInputParams params;
		int ret;
		params.inGame = true;
		ret = sys_memory_container_create(&m_VoucherInputDlgMemContainer, SCE_NP_COMMERCE2_DO_PRODUCT_CODE_MEMORY_CONTAINER_SIZE);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		params.memContainer = &m_VoucherInputDlgMemContainer;
		ret = Commerce::Interface::voucherCodeInput(params, false);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		return m_LastResult.GetResult();
	}

	// Gets category information that is available in the PlayStation®Store.
	// categoryId -	The category that information is being requested for. Specify NULL to receive information for the root category.
	ErrorCode CachedCommerce::RequestCategoryInfo(const char* categoryId)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(categoryId && (strlen(categoryId) == 0))
		{
			categoryId = NULL;
		}

		int ret = Commerce::Interface::getCategoryInfo(&m_CategoryInfo, categoryId, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}


	ErrorCode CachedCommerce::GetCategoryInfo(CommerceCategoryInfo* categoryInfo)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		CategoryInfoSub& category = m_CategoryInfo.get()->current;
		categoryInfo->categoryId = category.categoryId;
		categoryInfo->categoryName = category.categoryName;
		categoryInfo->categoryDescription = category.categoryDescription;
		categoryInfo->imageUrl = category.imageUrl;
		categoryInfo->countOfProducts = m_CategoryInfo.get()->countOfProducts;
		categoryInfo->countOfSubCategories = m_CategoryInfo.get()->countOfSubCategories;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::GetSubCategoryInfo(UInt32 index, CommerceCategoryInfo* categoryInfo)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(index >= m_CategoryInfo.get()->countOfSubCategories)
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		std::list<CategoryInfoSub>::iterator it = m_CategoryInfo.get()->subCategories.begin();
		if(index > 0)
		{
			std::advance(it, index);
		}
		CategoryInfoSub& category = *it;
		categoryInfo->categoryId = category.categoryId;
		categoryInfo->categoryName = category.categoryName;
		categoryInfo->categoryDescription = category.categoryDescription;
		categoryInfo->imageUrl = category.imageUrl;
		categoryInfo->countOfProducts = 0;
		categoryInfo->countOfSubCategories = 0;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::RequestProductList(const char* categoryId)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(categoryId && (strlen(categoryId) == 0))
		{
			categoryId = NULL;
		}

		int ret = Commerce::Interface::getProductList(&m_ProductList, categoryId, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}
	
	int CachedCommerce::GetProductListInfoCount()
	{
		if(IsBusy())
		{
			m_LastResult.SetResult(NP_ERR_BUSY, true);
			return 0;
		}
		SimpleLock::AutoLock lock(m_Lock);

		return m_ProductList.get() ? m_ProductList.get()->size() : 0;
	}

	ErrorCode CachedCommerce::GetProductListInfoItem(UInt32 index, CommerceProductInfo* productInfo)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(index > m_ProductList.get()->size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		std::vector<sce::Toolkit::NP::ProductInfo>& products = *m_ProductList.get();
		ProductInfo& info = products[index];
		productInfo->purchasabilityFlag = info.purchasabilityFlag;
		productInfo->productId = info.productId;
		productInfo->productName = info.productName;
		productInfo->shortDescription = info.shortDescription;
		productInfo->spName = info.spName;
		productInfo->imageUrl = info.imageUrl;
		productInfo->price = info.price;
		productInfo->releaseDate = (UInt64)info.releaseDate.tick;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::RequestDetailedProductInfo(const char* productId)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int ret = Commerce::Interface::getDetailedProductInfo(&m_ProductInfo, productId, NULL, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}
	
	ErrorCode CachedCommerce::GetProductInfoDetailed(CommerceProductInfoDetailed* productInfo)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		const ProductInfoDetailed& info = *m_ProductInfo.get();
		productInfo->purchasabilityFlag = info.purchasabilityFlag;
		productInfo->skuId = info.skuId;
		productInfo->productId = info.productId;
		productInfo->productName = info.productName;
		productInfo->shortDescription = info.shortDescription;
		productInfo->longDescription = info.longDescription;
		productInfo->legalDescription = info.legalDescription;
		productInfo->spName = info.spName;
		productInfo->imageUrl = info.imageUrl;
		productInfo->price = info.price;
		productInfo->ratingSystemId = info.ratingSystemId;
		productInfo->ratingImageUrl = info.ratingImageUrl;
		productInfo->releaseDate = (UInt64)info.releaseDate.tick;
		productInfo->numRatingDescriptors = info.ratingDescriptors.size();

		return m_LastResult.GetResult();
	}

#if(0)
	ErrorCode CachedCommerce::GetProductInfoRatingsDescriptor(UInt32 index)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		ProductInfoDetailed& info = *m_ProductInfo.get();
		if(index >= info.ratingDescriptors.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		std::list<SceNpCommerce2ContentRatingDescriptor>::iterator it = info.ratingDescriptors.begin();
		if(index > 0)
		{
			std::advance(it, index);
		}
		SceNpCommerce2ContentRatingDescriptor& desc = *it;
		// TODO: Need to understand how the rating descriptor is used.
		//	std::list<SceNpCommerce2ContentRatingDescriptor>::iterator iter = m_ProductInfo.get()->ratingDescriptors.begin();
		//	std::list<SceNpCommerce2ContentRatingDescriptor>::iterator iterEnd = m_ProductInfo.get()->ratingDescriptors.end();
		//	while (iter != iterEnd)
		//	{
		//		if (iter->descriptorType == SCE_NP_COMMERCE2_CONTENT_RATING_DESC_TYPE_ICON)
		//		{
		//			printf("Rating descriptor URL: %s\n", iter->imageUrl);
		//		}
		//		else if (iter->descriptorType == SCE_NP_COMMERCE2_CONTENT_RATING_DESC_TYPE_TEXT)
		//		{
		//			printf("Rating descriptor text: %s\n", iter->contentRatingDescription);
		//		}
		//		iter++;
		//	}
		//}

		return m_LastResult.GetResult();
	}
#endif

	ErrorCode CachedCommerce::RequestEntitlementList()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		// Make the currently cached list is reset.
		if(m_EntitlmentList.hasResult())
		{
			m_EntitlmentList.get()->clear();
		}

		int ret = Commerce::Interface::getEntitlementList(&m_EntitlmentList);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	int CachedCommerce::GetEntitlementListCount()
	{
		if(IsBusy())
		{
			m_LastResult.SetResult(NP_ERR_BUSY, true);
			return 0;
		}
		SimpleLock::AutoLock lock(m_Lock);

		return m_EntitlmentList.get() ? m_EntitlmentList.get()->size() : 0;
	}

	ErrorCode CachedCommerce::GetEntitlementListItem(UInt32 index, CommerceEntitlement* entitlement)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(index > m_EntitlmentList.get()->size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		std::vector<SceNpEntitlement>& ents = *m_EntitlmentList.get();
		SceNpEntitlement& info = ents[index];
		entitlement->id = (const char*)&info.id.data[0];
		entitlement->createdDate = info.created_date;
		entitlement->expireDate = info.expire_date;
		entitlement->type = info.type;
		entitlement->remainingCount = info.remaining_count;
		entitlement->consumedCount = info.consumed_count;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedCommerce::ConsumeEntitlement(const char* entitlementId, int consumeCount)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		SceNpEntitlementId entId;
		memset(&entId, 0x0, sizeof(SceNpEntitlementId));
		strcpy((char*)entId.data, entitlementId);
		int ret = Commerce::Interface::consumeEntitlement(entId, consumeCount, true);

		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		
		m_Busy = true;
		return m_LastResult.GetResult();
	}

	void CachedCommerce::CleanupDialogMem()
	{
		if(m_CheckoutDlgMemContainer != SYS_MEMORY_CONTAINER_ID_INVALID)
		{
			sys_memory_container_destroy(m_CheckoutDlgMemContainer);
			m_CheckoutDlgMemContainer = SYS_MEMORY_CONTAINER_ID_INVALID;
			DBG_LOG("Commerce: released checkout dialog mem");
		}
		if(m_DlListDlgMemContainer != SYS_MEMORY_CONTAINER_ID_INVALID)
		{
			sys_memory_container_destroy(m_DlListDlgMemContainer);
			m_DlListDlgMemContainer = SYS_MEMORY_CONTAINER_ID_INVALID;
			DBG_LOG("Commerce: released download list dialog mem");
		}
		if(m_BrowseProductDlgMemContainer != SYS_MEMORY_CONTAINER_ID_INVALID)
		{
			sys_memory_container_destroy(m_BrowseProductDlgMemContainer);
			m_BrowseProductDlgMemContainer = SYS_MEMORY_CONTAINER_ID_INVALID;
			DBG_LOG("Commerce: released product dialog mem");
		}
		if(m_VoucherInputDlgMemContainer != SYS_MEMORY_CONTAINER_ID_INVALID)
		{
			sys_memory_container_destroy(m_VoucherInputDlgMemContainer);
			m_VoucherInputDlgMemContainer = SYS_MEMORY_CONTAINER_ID_INVALID;
			DBG_LOG("Commerce: released voucher dialog mem");
		}
	}

}
